/* FILE: mscrnhdr.c                             (D. Tottingham  03/24/91)

This is a collection of C helper functions that manage the status header
information that is kept on the top of the monochrome display for
xdetect.  All functions have been written and compiled medium model.  The
following functions are included:

h_increment_bankswitch_ctr ()   increment bank switch counter
h_increment_event_ctr ()        increment event counter
h_initialize_params ()          initialize a screen header
h_initialize_path ()            initialize path_name
h_set_event_ctr ()              set event counter
h_set_trigger_status ()         set trigger status
h_set_uptime ()                 update uptime.
h_toggle_status ()              toggle a status bit
h_update ()                     update the header

EXTERNAL FUNCTIONS CALLED:

dm_get_uptime ()                get session up time.
u_get_diskspace ()              get the percentage of free disk space
u_gmtime ()                     convert time value into structure.

HISTORY:
   none


/*************************************************************************
                             INCLUDE FILES


*************************************************************************/
#include <stdio.h>
#include <string.h>

#include "mconst.h"
#include "mdemux.h"
#include "mscrnhdr.h"
#include "mutils.h"
#include "mvideo.h"
#include "xdetect.h"


/*************************************************************************
                              GLOBAL DATA


**************************************************************************/
PRIVATE H_STATUS header, *h = &header;
PRIVATE char path_name[MAX_FILENAME_LENGTH];


/*=======================================================================*
 *                          build_storage_stat                           *
 *=======================================================================*/
/* Build a storage status string.  Percentage of free space is converted
   converted into an ascii string.                                       */

PRIVATE
void build_storage_stat (storage_status)
char * storage_status;
{
   static char * device_id[] = {"A:","B:","C:","D:","E:","F:","G:",NULL};
   long percent_avail;
   int drive;

   percent_avail = u_get_diskspace (path_name, &drive);

   sprintf (storage_status, "  %2s%3ld%%", device_id[drive], percent_avail);
}

/*=======================================================================*
 *                     h_increment_bankswitch_ctr                        *
 *=======================================================================*/
/* Increment bank switch counter.                                        */

PUBLIC
void h_increment_bankswitch_ctr ()
{
   h->bankswitch_ctr++;
   h_update();
}

/*=======================================================================*
 *                        h_increment_event_ctr                          *
 *=======================================================================*/
/* Increment event counter.                                              */

PUBLIC
void h_increment_event_ctr ()
{
   h->event_ctr++;
   h_update();
}

/*=======================================================================*
 *                         h_initialize_params                           *
 *=======================================================================*/
/* Initialize a screen header.                                           */

PUBLIC
void h_initialize_params ()
{
   h->milestone = REVISION_NAME;
   h->event_status = H_ASLEEP;
   h->event_ctr = 0;
   h->uptime.days = 0;
   h->uptime.hours = 0;
   h->uptime.minutes = 0;
   h->uptime.seconds = 0;
   h->uptime.millisec = 0;
}

/*=======================================================================*
 *                          h_initialize_path                            *
 *=======================================================================*/
/* Initialize path_name.                                                 */

PUBLIC
void h_initialize_path (pathname)
char pathname[];
{
   strcpy (path_name, pathname);
}

/*=======================================================================*
 *                            h_set_event_ctr                            *
 *=======================================================================*/
/* Set event counter.                                                    */

PUBLIC
void h_set_event_ctr (event_ctr)
int event_ctr;
{
   h->event_ctr = event_ctr;
   h_update();
}

/*=======================================================================*
 *                          h_set_trigger_status                         *
 *=======================================================================*/
/* Set trigger status.                                                   */

PUBLIC
void h_set_trigger_status (trigger_status)
int trigger_status;
{
   if (trigger_status)
      h->event_status |= H_AUTOTRIGGER_ENABLED;
   else
      h->event_status &= ~H_AUTOTRIGGER_ENABLED;
}

/*=======================================================================*
 *                             h_toggle_status                           *
 *=======================================================================*/
/* Toggle a status bit.                                                  */

PUBLIC
void h_toggle_status (status_bit)
int status_bit;
{
   if (h->event_status & status_bit)
      /* Bit is on so turn it off */
      h->event_status &= ~status_bit;
   else
      h->event_status |= status_bit;
   h_update();
}

/*=======================================================================*
 *                                h_update                               *
 *=======================================================================*/
/* Updates the status header information on monochrome display.          */

PUBLIC
void h_update ()
{
   static char * event_status[] = {             /* status conversion array */
                  "Recording Event",                            /* 0     */
                  "Recording Calibration",                      /* 1     */
                  "Free Run Enabled",                           /* 2     */
                  "Autotriggering",                             /* 3     */
                  "Asleep"                                      /* 4     */
                  };

   static char top_hdr[90], right_hdr[70];
   unsigned short bit_ptr;
   int i, j;

   /* Get the current storage status */
   build_storage_stat (h->storage_status);

   /* Prepare right header string */
   for (bit_ptr = 1, i = 0; !(h->event_status & bit_ptr); bit_ptr <<= 1, i++);

   j = sprintf (right_hdr, "%-7s free ", h->storage_status);
   if (h->event_ctr == 0)
      j += sprintf (right_hdr+j, " No events");
   else if (h->event_ctr == 1)
      j += sprintf (right_hdr+j, "  1 event ");
   else j += sprintf (right_hdr+j, "%3d events", h->event_ctr);

   if (h->bankswitch_ctr == 0)
      j += sprintf (right_hdr+j, "        ");
   else
      j += sprintf (right_hdr+j, "/ %3d bs", h->bankswitch_ctr);

   j += sprintf (right_hdr+j, "%23s", event_status[i]);

   /* Prepare top header string */
   sprintf (top_hdr, "%-10.10s F1=help %60s", h->milestone, right_hdr);

   /* Write header strings on screen */
   gputs ( HDR1_I1, 4, top_hdr, 90, HEADER_COLORS);

   /* Prepare uptime string */
   sprintf (top_hdr, "Up %3d days, %02d:%02d:%02d", h->uptime.days,
            h->uptime.hours, h->uptime.minutes, h->uptime.seconds);

   /* Write uptime on next line */
   gputs ( HDR1_I2, 4, top_hdr, 90, HEADER_COLORS);
}

/*=======================================================================*
 *                              h_set_uptime                             *
 *=======================================================================*/

PUBLIC
void h_set_uptime ()
{
   u_gmtime (dm_get_uptime(), &(h->uptime));
   h_update();
}
